#!/bin/bash

TAR="$1"

if [[ ! -f "$TAR" ]]; then
    echo "❌ No se encontró ningún archivo .tar"
    exit 1
fi

BASE=$(basename "$TAR" .tar)
IMAGEN_ORIGEN=""
CONTENEDOR="${BASE}_ctn"
IMAGEN_NUEVA="${BASE}_modificada:latest"

function salir() {
    echo -e "\n🛑 Ctrl+C detectado. Guardando cambios..."

    echo "🧱 Creando nueva imagen $IMAGEN_NUEVA desde contenedor $CONTENEDOR..."
    docker commit "$CONTENEDOR" "$IMAGEN_NUEVA"

    # Preguntar si quiere exportar a tar
    read -rp "¿Querés exportar la imagen a un archivo .tar? (s/n): " RESP
    if [[ "$RESP" =~ ^[Ss]$ ]]; then
        read -rp "Ingresá el nombre del archivo .tar (sin extensión): " NOMBRE_TAR
        NOMBRE_TAR="${NOMBRE_TAR}.tar"
        echo "💾 Exportando imagen a $NOMBRE_TAR ..."
        docker save -o "$NOMBRE_TAR" "$IMAGEN_NUEVA"
        echo "✅ Imagen exportada a $NOMBRE_TAR"
    else
        echo "⚠️ No se exportó la imagen a archivo .tar"
    fi

    echo "🛑 Deteniendo y eliminando contenedor $CONTENEDOR..."
    docker stop "$CONTENEDOR" >/dev/null 2>&1
    docker rm "$CONTENEDOR" >/dev/null 2>&1

    # Preguntar si quiere limpiar imágenes no utilizadas
    read -rp "¿Querés eliminar imágenes no utilizadas? (docker image prune -a) (s/n): " LIMPIAR_IMG
    if [[ "$LIMPIAR_IMG" =~ ^[Ss]$ ]]; then
        echo "🧹 Limpiando imágenes no utilizadas..."
        docker image prune -a -f
        echo "✅ Imágenes no utilizadas eliminadas."
    else
        echo "🔒 Las imágenes no utilizadas no fueron eliminadas."
    fi

    # Preguntar si quiere eliminar volúmenes no usados
    read -rp "¿Querés eliminar volúmenes no utilizados? (docker volume prune) (s/n): " LIMPIAR_VOL
    if [[ "$LIMPIAR_VOL" =~ ^[Ss]$ ]]; then
        echo "🧹 Limpiando volúmenes no utilizados..."
        docker volume prune -f
        echo "✅ Volúmenes no utilizados eliminados."
    else
        echo "🔒 Los volúmenes no utilizados no fueron eliminados."
    fi

    echo "✅ Contenedor detenido y eliminado. Saliendo."
    exit 0
}

trap salir INT

echo "📦 Cargando imagen desde $TAR..."
OUTPUT=$(docker load -i "$TAR")
echo "$OUTPUT"

IMAGEN_ORIGEN=$(echo "$OUTPUT" | grep "Loaded image:" | awk -F': ' '{print $2}')
if [[ -z "$IMAGEN_ORIGEN" ]]; then
    # Buscar imagen cargada por si ya existía
    IMAGEN_ORIGEN=$(docker images --format "{{.Repository}}:{{.Tag}}" | grep "^$BASE")
fi

if [[ -z "$IMAGEN_ORIGEN" ]]; then
    echo "❌ No se encontró la imagen $BASE cargada."
    exit 1
fi

if docker ps -a --format '{{.Names}}' | grep -Eq "^$CONTENEDOR\$"; then
    echo "🗑️ Eliminando contenedor anterior $CONTENEDOR..."
    docker rm -f "$CONTENEDOR"
fi

echo "🚀 Iniciando contenedor '$CONTENEDOR'..."
docker run -d --name "$CONTENEDOR" \
  -p 1022:22 \
  -p 1021:21 \
  -p 1080:80 \
  "$IMAGEN_ORIGEN"

sleep 2
IP=$(docker inspect -f '{{range .NetworkSettings.Networks}}{{.IPAddress}}{{end}}' "$CONTENEDOR")

echo "✅ Contenedor '$CONTENEDOR' corriendo"
echo "🌐 IP interna: $IP"
echo "🛑 Presioná Ctrl+C para guardar cambios y detener."

while true; do
    sleep 1
done
