#!/bin/bash

# ==========================
# COLORES
# ==========================
R="\e[31m"; G="\e[32m"; Y="\e[33m"; B="\e[34m"; N="\e[0m"

# ==========================
# CONFIG
# ==========================
TAR="$1"
BASE=""
CONTENEDOR=""
IMAGEN_ORIGEN=""
IMAGEN_NUEVA=""

# ==========================
# FUNCIONES
# ==========================
puerto_libre() {
    for p in $(seq $1 $2); do
        ss -tuln | awk '{print $5}' | grep -q ":$p$" || { echo "$p"; return; }
    done
}

banner() {
    clear
    echo -e "${R}"
    echo " ███████╗     ██╗██████╗ "
    echo " ██╔════╝     ██║██╔══██╗"
    echo " ███████╗     ██║██║  ██║"
    echo " ╚════██║██   ██║██║  ██║"
    echo " ███████║╚█████╔╝██████╔╝"
    echo " ╚══════╝ ╚════╝ ╚═════╝ "
    echo -e "${N}"
}

salir() {
    echo -e "\n${Y}🛑 Ctrl+C detectado${N}"

    if docker ps --format '{{.Names}}' | grep -q "^$CONTENEDOR$"; then
        echo -e "${B}📸 Creando snapshot...${N}"
        docker commit "$CONTENEDOR" "$IMAGEN_NUEVA" >/dev/null
        echo -e "${G}✅ Imagen creada: $IMAGEN_NUEVA${N}"
    fi

    read -rp "📦 ¿Exportar imagen a .tar? (s/n): " RESP
    if [[ "$RESP" =~ ^[Ss]$ ]]; then
        docker save -o "${IMAGEN_NUEVA}.tar" "$IMAGEN_NUEVA"
        sha256sum "${IMAGEN_NUEVA}.tar" > "${IMAGEN_NUEVA}.sha256"
        echo -e "${G}✔ Exportado + checksum${N}"
    fi

    echo -e "${Y}🗑 Eliminando contenedor...${N}"
    docker rm -f "$CONTENEDOR" >/dev/null 2>&1

    read -rp "🧹 ¿Limpiar imágenes no usadas? (s/n): " LIMPIAR
    [[ "$LIMPIAR" =~ ^[Ss]$ ]] && docker image prune -a -f

    echo -e "${G}👋 Fin.${N}"
    exit 0
}

trap salir INT

# ==========================
# VALIDACIÓN
# ==========================
[[ -z "$TAR" ]] && echo "Uso: $0 imagen.tar" && exit 1
[[ ! -f "$TAR" ]] && echo "❌ No existe $TAR" && exit 1

banner

# ==========================
# CARGAR IMAGEN
# ==========================
echo -e "${B}📦 Cargando imagen...${N}"
OUT=$(docker load -i "$TAR")
IMAGEN_ORIGEN=$(echo "$OUT" | awk -F': ' '/Loaded image:/ {print $2}')

BASE=$(basename "$IMAGEN_ORIGEN" | cut -d: -f1)
CONTENEDOR="${BASE}_ctn"
IMAGEN_NUEVA="${BASE}_snapshot:latest"

# ==========================
# PUERTOS AUTOMÁTICOS
# ==========================
P_SSH=$(puerto_libre 2200 2299)
P_FTP=$(puerto_libre 2100 2199)
P_WEB=$(puerto_libre 8000 8099)

echo -e "${G}🔑 SSH → $P_SSH${N}"
echo -e "${G}📂 FTP → $P_FTP${N}"
echo -e "${G}🌍 WEB → $P_WEB${N}"

# ==========================
# LIMPIAR CONTENEDOR
# ==========================
docker rm -f "$CONTENEDOR" >/dev/null 2>&1

# ==========================
# RUN
# ==========================
echo -e "${B}🚀 Lanzando contenedor...${N}"
docker run -d --name "$CONTENEDOR" \
  -p $P_SSH:22 \
  -p $P_FTP:21 \
  -p $P_WEB:80 \
  "$IMAGEN_ORIGEN"

sleep 2
IP=$(docker inspect -f '{{range .NetworkSettings.Networks}}{{.IPAddress}}{{end}}' "$CONTENEDOR")

# ==========================
# README AUTO
# ==========================
cat <<EOF > README_${BASE}.md
# Máquina Vulnerable: $BASE

## Accesos
- SSH: localhost:$P_SSH
- FTP: localhost:$P_FTP
- WEB: http://localhost:$P_WEB
- IP interna: $IP

## Imagen base
$IMAGEN_ORIGEN

## Hash
\`\`\`
sha256sum ${IMAGEN_NUEVA}.tar
\`\`\`
EOF

# ==========================
# INFO FINAL
# ==========================
echo
echo -e "${G}====================================${N}"
echo -e "${G} Máquina LISTA${N}"
echo -e "${G} Contenedor : $CONTENEDOR${N}"
echo -e "${G} IP         : $IP${N}"
echo -e "${G}====================================${N}"
echo -e "${Y}🛑 Ctrl+C para snapshot y exportar${N}"

while true; do sleep 1; done
